<?php

namespace App\Http\Controllers;

use App\Models\Sale;
use App\Models\Purchase;
use App\Models\Product;
use App\Models\SaleItem;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index()
    {
        return view('reports.index');
    }

    public function sales(Request $request)
    {
        $dateFrom = $request->input('date_from', now()->startOfMonth());
        $dateTo = $request->input('date_to', now()->endOfMonth());

        $sales = Sale::whereBetween('created_at', [$dateFrom, $dateTo])
            ->selectRaw('DATE(created_at) as date')
            ->selectRaw('COUNT(*) as count')
            ->selectRaw('SUM(total_amount) as total')
            ->selectRaw('SUM(discount) as discount')
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->get();

        $totalSales = $sales->sum('total');
        $totalTransactions = $sales->sum('count');
        $totalDiscount = $sales->sum('discount');

        return view('reports.sales', compact(
            'sales',
            'totalSales',
            'totalTransactions',
            'totalDiscount',
            'dateFrom',
            'dateTo'
        ));
    }

    public function purchases(Request $request)
    {
        $dateFrom = $request->input('date_from', now()->startOfMonth());
        $dateTo = $request->input('date_to', now()->endOfMonth());

        $purchases = Purchase::with('supplier')
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->get();

        $totalPurchases = $purchases->sum('total_amount');
        $totalPaid = $purchases->sum('paid_amount');
        $totalRemaining = $totalPurchases - $totalPaid;

        return view('reports.purchases', compact(
            'purchases',
            'totalPurchases',
            'totalPaid',
            'totalRemaining',
            'dateFrom',
            'dateTo'
        ));
    }

    public function profit(Request $request)
    {
        $dateFrom = $request->input('date_from', now()->startOfMonth());
        $dateTo = $request->input('date_to', now()->endOfMonth());

        $salesData = SaleItem::join('sales', 'sale_items.sale_id', '=', 'sales.id')
            ->join('products', 'sale_items.product_id', '=', 'products.id')
            ->whereBetween('sales.created_at', [$dateFrom, $dateTo])
            ->select(
                DB::raw('SUM(sale_items.total_price) as revenue'),
                DB::raw('SUM(sale_items.quantity * products.cost_price) as cost')
            )
            ->first();

        $revenue = $salesData->revenue ?? 0;
        $cost = $salesData->cost ?? 0;
        $profit = $revenue - $cost;
        $profitMargin = $revenue > 0 ? ($profit / $revenue) * 100 : 0;

        return view('reports.profit', compact(
            'revenue',
            'cost',
            'profit',
            'profitMargin',
            'dateFrom',
            'dateTo'
        ));
    }

    public function inventory()
    {
        $products = Product::with('category')
            ->select('products.*')
            ->selectRaw('(selling_price - cost_price) * quantity as potential_profit')
            ->selectRaw('cost_price * quantity as inventory_value')
            ->get();

        $totalValue = $products->sum('inventory_value');
        $totalPotentialProfit = $products->sum('potential_profit');
        $lowStockCount = $products->filter(fn($p) => $p->quantity <= $p->min_quantity)->count();

        return view('reports.inventory', compact(
            'products',
            'totalValue',
            'totalPotentialProfit',
            'lowStockCount'
        ));
    }

    public function topProducts(Request $request)
    {
        $dateFrom = $request->input('date_from', now()->startOfMonth());
        $dateTo = $request->input('date_to', now()->endOfMonth());

        $products = Product::join('sale_items', 'products.id', '=', 'sale_items.product_id')
            ->join('sales', 'sale_items.sale_id', '=', 'sales.id')
            ->whereBetween('sales.created_at', [$dateFrom, $dateTo])
            ->select(
                'products.*',
                DB::raw('SUM(sale_items.quantity) as total_quantity'),
                DB::raw('SUM(sale_items.total_price) as total_revenue'),
                DB::raw('COUNT(DISTINCT sales.id) as times_sold')
            )
            ->groupBy('products.id')
            ->orderBy('total_revenue', 'desc')
            ->take(20)
            ->get();

        return view('reports.top-products', compact('products', 'dateFrom', 'dateTo'));
    }
}
