{{-- resources/views/pos/index.blade.php --}}
@extends('layouts.app')

@section('page-title', 'نقطة البيع')

@section('styles')
    <style>
        .pos-container { background: #f8f9fa; }
        .product-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            gap: 15px;
        }
        .product-card {
            background: white;
            border-radius: 10px;
            padding: 15px;
            cursor: pointer;
            transition: all 0.3s;
            border: 2px solid transparent;
        }
        .product-card:hover {
            border-color: #3498db;
            transform: scale(1.05);
        }
        .product-card.out-of-stock {
            opacity: 0.5;
            cursor: not-allowed;
        }
        .cart-section {
            background: white;
            border-radius: 10px;
            padding: 20px;
            height: calc(100vh - 200px);
            display: flex;
            flex-direction: column;
        }
        .cart-items {
            flex: 1;
            overflow-y: auto;
            max-height: 400px;
        }
        .cart-item {
            border-bottom: 1px solid #eee;
            padding: 10px 0;
        }
        .total-section {
            border-top: 2px solid #eee;
            padding-top: 15px;
            margin-top: 15px;
        }
        .barcode-input {
            font-size: 1.2rem;
            padding: 15px;
        }
    </style>
@endsection

@section('content')
    <div class="pos-container">
        <div class="row g-3">
            <!-- Products Section -->
            <div class="col-lg-6">
                <div class="card mb-3">
                    <div class="card-body">
                        <div class="row g-2">
                            <div class="col-md-8">
                                <input type="text"
                                       id="barcodeInput"
                                       class="form-control barcode-input"
                                       placeholder="امسح الباركود أو ابحث عن منتج..."
                                       autofocus>
                            </div>
                            <div class="col-md-4">
                                <select class="form-select" id="categoryFilter">
                                    <option value="">كل الفئات</option>
                                    @foreach($products->unique('category_id') as $product)
                                        <option value="{{ $product->category_id }}">{{ $product->category->name }}</option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-body" style="max-height: calc(100vh - 300px); overflow-y: auto;">
                        <div class="product-grid" id="productGrid">
                            @foreach($products as $product)
                                <div class="product-card {{ $product->quantity <= 0 ? 'out-of-stock' : '' }}"
                                     data-product-id="{{ $product->id }}"
                                     data-product-name="{{ $product->name }}"
                                     data-product-price="{{ $product->selling_price }}"
                                     data-product-quantity="{{ $product->quantity }}"
                                     data-product-barcode="{{ $product->barcode }}"
                                     data-category="{{ $product->category_id }}">
                                    <div class="text-center">
                                        @if($product->image)
                                            <img src="{{ asset('storage/' . $product->image) }}"
                                                 alt="{{ $product->name }}"
                                                 style="width: 80px; height: 80px; object-fit: cover; border-radius: 8px;">
                                        @else
                                            <div class="bg-light rounded" style="width: 80px; height: 80px; margin: 0 auto; display: flex; align-items: center; justify-content: center;">
                                                <i class="bi bi-box-seam fs-1 text-muted"></i>
                                            </div>
                                        @endif
                                        <p class="mt-2 mb-1 fw-bold small">{{ $product->name }}</p>
                                        <p class="text-primary mb-0 small">{{ number_format($product->selling_price, 2) }} ج.م</p>
                                        <small class="text-muted">المخزون: {{ $product->quantity }}</small>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    </div>
                </div>
            </div>

            <!-- Cart Section -->
            <div class="col-lg-6">
                <div class="cart-section">
                    <h5 class="mb-3"><i class="bi bi-cart"></i> سلة المشتريات</h5>

                    <div class="cart-items" id="cartItems">
                        <div class="text-center text-muted py-5">
                            <i class="bi bi-cart3 fs-1"></i>
                            <p class="mt-2">السلة فارغة</p>
                        </div>
                    </div>

                    <div class="total-section">
                        <div class="d-flex justify-content-between mb-2">
                            <span>المجموع الفرعي:</span>
                            <strong id="subtotal">0.00 ج.م</strong>
                        </div>
                        <div class="d-flex justify-content-between mb-2">
                            <span>الخصم:</span>
                            <input type="number"
                                   id="discount"
                                   class="form-control form-control-sm"
                                   value="0"
                                   min="0"
                                   style="width: 100px; text-align: center;">
                        </div>
                        <div class="d-flex justify-content-between mb-3">
                            <h5>الإجمالي:</h5>
                            <h5 class="text-primary" id="total">0.00 ج.م</h5>
                        </div>

                        <div class="mb-3">
                            <label>طريقة الدفع:</label>
                            <select class="form-select" id="paymentMethod">
                                <option value="cash">نقدي</option>
                                <option value="card">بطاقة</option>
                                <option value="mobile_wallet">محفظة إلكترونية</option>
                            </select>
                        </div>

                        <div class="d-grid gap-2">
                            <button class="btn btn-success btn-lg" id="checkoutBtn">
                                <i class="bi bi-check-circle"></i> إتمام البيع (F2)
                            </button>
                            <button class="btn btn-danger" id="clearCartBtn">
                                <i class="bi bi-x-circle"></i> إلغاء
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Payment Modal -->
    <div class="modal fade" id="paymentModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-success text-white">
                    <h5 class="modal-title">إتمام عملية الدفع</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">المبلغ الإجمالي:</label>
                        <input type="text" class="form-control form-control-lg text-center" id="modalTotal" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">المبلغ المدفوع:</label>
                        <input type="number" class="form-control form-control-lg text-center" id="paidAmount" min="0" step="0.01">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الباقي:</label>
                        <input type="text" class="form-control form-control-lg text-center text-success" id="changeAmount" readonly>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="button" class="btn btn-success" id="completeSaleBtn">تأكيد البيع</button>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('scripts')
    <script>
        // Global variables
        let cart = [];
        const products = @json($products);

        console.log('Products loaded:', products.length);

        // Wait for document ready
        $(document).ready(function() {
            console.log('POS System initialized');

            // Event delegation for product cards - click to add to cart
            $(document).on('click', '.product-card', function(e) {
                e.preventDefault();
                const productId = parseInt($(this).data('product-id'));
                console.log('Product clicked:', productId);

                if (productId && !$(this).hasClass('out-of-stock')) {
                    addToCart(productId);
                } else {
                    console.log('Product out of stock or invalid');
                }
            });

            // Barcode scanner - Enter key
            $('#barcodeInput').on('keypress', function(e) {
                if (e.which === 13) { // Enter key
                    const barcode = $(this).val().trim();
                    console.log('Barcode scanned:', barcode);

                    if (barcode) {
                        const product = products.find(p => p.barcode === barcode);
                        if (product) {
                            addToCart(product.id);
                            $(this).val('');
                        } else {
                            alert('المنتج غير موجود!');
                        }
                    }
                }
            });

            // Search products as you type
            $('#barcodeInput').on('input', function() {
                const search = $(this).val().toLowerCase();
                console.log('Searching for:', search);

                $('.product-card').each(function() {
                    const name = $(this).data('product-name').toString().toLowerCase();
                    const barcode = $(this).data('product-barcode').toString();
                    const match = name.includes(search) || barcode.includes(search);
                    $(this).toggle(match);
                });
            });

            // Category filter
            $('#categoryFilter').on('change', function() {
                const categoryId = $(this).val();
                console.log('Filter by category:', categoryId);

                $('.product-card').each(function() {
                    if (!categoryId || $(this).data('category') == categoryId) {
                        $(this).show();
                    } else {
                        $(this).hide();
                    }
                });
            });

            // Event listeners for cart buttons
            $('#checkoutBtn').on('click', checkout);
            $('#clearCartBtn').on('click', clearCart);
            $('#completeSaleBtn').on('click', completeSale);
            $('#discount').on('change keyup', updateTotals);

            // Event delegation for cart item buttons
            $(document).on('click', '.remove-item', function() {
                const index = $(this).data('index');
                removeFromCart(index);
            });

            $(document).on('click', '.qty-minus', function() {
                const index = $(this).data('index');
                updateQuantity(index, cart[index].quantity - 1);
            });

            $(document).on('click', '.qty-plus', function() {
                const index = $(this).data('index');
                updateQuantity(index, cart[index].quantity + 1);
            });

            $(document).on('change', '.qty-input', function() {
                const index = $(this).data('index');
                const quantity = parseInt($(this).val());
                updateQuantity(index, quantity);
            });

            // Payment modal input
            $('#paidAmount').on('input', function() {
                const total = parseFloat($('#total').text());
                const paid = parseFloat($(this).val()) || 0;
                const change = paid - total;
                $('#changeAmount').val(change.toFixed(2) + ' ج.م');
            });

            // Keyboard shortcuts
            $(document).keydown(function(e) {
                if (e.key === 'F2') {
                    e.preventDefault();
                    checkout();
                }
            });
        });

        // Add product to cart
        function addToCart(productId) {
            console.log('Adding to cart:', productId);
            const product = products.find(p => p.id === productId);

            if (!product || product.quantity <= 0) {
                alert('المنتج غير متوفر!');
                return;
            }

            const existingItem = cart.find(item => item.product_id === productId);

            if (existingItem) {
                if (existingItem.quantity < product.quantity) {
                    existingItem.quantity++;
                    console.log('Increased quantity:', existingItem.quantity);
                } else {
                    alert('الكمية المطلوبة غير متوفرة!');
                    return;
                }
            } else {
                cart.push({
                    product_id: productId,
                    name: product.name,
                    unit_price: parseFloat(product.selling_price),
                    quantity: 1,
                    max_quantity: product.quantity
                });
                console.log('Added new item to cart');
            }

            renderCart();
            updateTotals();

            // Focus back to barcode input
            $('#barcodeInput').focus();
        }

        // Remove item from cart
        function removeFromCart(index) {
            console.log('Removing item at index:', index);
            cart.splice(index, 1);
            renderCart();
            updateTotals();
        }

        // Update item quantity
        function updateQuantity(index, quantity) {
            quantity = parseInt(quantity);
            console.log('Updating quantity:', index, quantity);

            if (quantity <= 0) {
                removeFromCart(index);
                return;
            }

            if (quantity > cart[index].max_quantity) {
                alert('الكمية المطلوبة غير متوفرة!');
                quantity = cart[index].max_quantity;
            }

            cart[index].quantity = quantity;
            renderCart();
            updateTotals();
        }

        // Render cart items
        function renderCart() {
            console.log('Rendering cart:', cart.length, 'items');

            const cartHtml = cart.length === 0
                ? '<div class="text-center text-muted py-5"><i class="bi bi-cart3 fs-1"></i><p class="mt-2">السلة فارغة</p></div>'
                : cart.map((item, index) => `
            <div class="cart-item">
                <div class="d-flex justify-content-between align-items-center mb-2">
                    <strong>${item.name}</strong>
                    <button class="btn btn-sm btn-danger remove-item" data-index="${index}">
                        <i class="bi bi-trash"></i>
                    </button>
                </div>
                <div class="d-flex justify-content-between align-items-center">
                    <div class="input-group" style="width: 120px;">
                        <button class="btn btn-outline-secondary btn-sm qty-minus" data-index="${index}">-</button>
                        <input type="number" class="form-control form-control-sm text-center qty-input" value="${item.quantity}"
                               data-index="${index}" min="1" max="${item.max_quantity}">
                        <button class="btn btn-outline-secondary btn-sm qty-plus" data-index="${index}">+</button>
                    </div>
                    <strong>${(item.quantity * item.unit_price).toFixed(2)} ج.م</strong>
                </div>
            </div>
        `).join('');

            $('#cartItems').html(cartHtml);
        }

        // Update totals
        function updateTotals() {
            const subtotal = cart.reduce((sum, item) => sum + (item.quantity * item.unit_price), 0);
            const discount = parseFloat($('#discount').val()) || 0;
            const total = subtotal - discount;

            $('#subtotal').text(subtotal.toFixed(2) + ' ج.م');
            $('#total').text(total.toFixed(2) + ' ج.م');

            console.log('Totals updated - Subtotal:', subtotal, 'Total:', total);
        }

        // Clear cart
        function clearCart() {
            if (confirm('هل تريد إلغاء عملية البيع؟')) {
                cart = [];
                renderCart();
                updateTotals();
                $('#discount').val(0);
                console.log('Cart cleared');
            }
        }

        // Checkout
        function checkout() {
            if (cart.length === 0) {
                alert('السلة فارغة!');
                return;
            }

            const total = parseFloat($('#total').text());
            $('#modalTotal').val(total.toFixed(2) + ' ج.م');
            $('#paidAmount').val(total.toFixed(2));
            $('#changeAmount').val('0.00 ج.م');

            const modal = new bootstrap.Modal(document.getElementById('paymentModal'));
            modal.show();

            console.log('Checkout initiated');
        }

        // Complete sale
        function completeSale() {
            const total = parseFloat($('#total').text());
            const paid = parseFloat($('#paidAmount').val()) || 0;

            if (paid < total) {
                alert('المبلغ المدفوع أقل من الإجمالي!');
                return;
            }

            console.log('Completing sale...');

            $.ajax({
                url: '{{ route("pos.store") }}',
                method: 'POST',
                data: {
                    _token: '{{ csrf_token() }}',
                    items: cart,
                    subtotal: parseFloat($('#subtotal').text()),
                    discount: parseFloat($('#discount').val()) || 0,
                    tax: 0,
                    total_amount: total,
                    payment_method: $('#paymentMethod').val(),
                    paid_amount: paid
                },
                success: function(response) {
                    console.log('Sale completed:', response);

                    bootstrap.Modal.getInstance(document.getElementById('paymentModal')).hide();

                    alert('تمت عملية البيع بنجاح!\nرقم الفاتورة: ' + response.invoice_number);

                    window.open('{{ url("/pos/receipt") }}/' + response.sale_id, '_blank');

                    // Clear cart
                    cart = [];
                    renderCart();
                    updateTotals();
                    $('#discount').val(0);
                },
                error: function(xhr) {
                    console.error('Sale error:', xhr);
                    alert('حدث خطأ: ' + (xhr.responseJSON?.error || 'خطأ غير معروف'));
                }
            });
        }
    </script>
@endsection
